﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Domain.Configurations;
    using Hims.Shared.EntityModels;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.Filters;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.Library;
    using Utilities;
    using TimeZone = Shared.Library.TimeZone;

    /// <inheritdoc />
    /// <summary>
    /// The resources controller.
    /// </summary>
    [Route("api/resources")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ResourcesController : BaseController
    {
        /// <summary>
        /// The resource services.
        /// </summary>
        private readonly IResourceService resourceServices;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The amazon s 3 configuration.
        /// </summary>
        private readonly IAmazonS3Configuration amazonS3Configuration;

        /// <summary>
        /// The appointments services.
        /// </summary>
        private readonly IAppointmentService appointmentsServices;

        /// <summary>
        /// The admission services.
        /// </summary>
        private readonly IAdmissionService admissionService;

        /// <summary>
        /// The FTP upload helper.
        /// </summary>
        private readonly IFtpUploadHelper ftpUploadHelper;

        /// <inheritdoc />
        public ResourcesController(IResourceService resourceServices, IAESHelper aesHelper, IAmazonS3Configuration amazonS3Configuration, IAppointmentService appointmentsServices, IAdmissionService admissionService, IFtpUploadHelper ftpUploadHelper)
        {
            this.resourceServices = resourceServices;
            this.aesHelper = aesHelper;
            this.amazonS3Configuration = amazonS3Configuration;
            this.appointmentsServices = appointmentsServices;
            this.admissionService = admissionService;
            this.ftpUploadHelper = ftpUploadHelper;
        }

        /// <summary>
        /// To fetch list of roles.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of roles.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("roles")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRolesAsync()
        {
            var resources = await this.resourceServices.RolesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of roles except patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of roles.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("rolesWithOutPatient")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> RolesWithOutPatient()
        {
            var resources = await this.resourceServices.RolesWithOutPatient();
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of countries.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("countries")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCountriesAsync()
        {
            try
            {
                var resources = await this.resourceServices.CountriesAsync();
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }


        /// <summary>
        /// To fetch list of countries.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("locations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationsAsync()
        {
            try
            {
                var resources = await this.resourceServices.LocationsAsync();
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }


        /// <summary>
        /// To fetch list of time zones.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of time zones.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("time-zones")]
        [ProducesResponseType(200, Type = typeof(List<TimeZone>))]
        [ProducesResponseType(500)]
        public ActionResult FetchTimeZonesAsync()
        {
            var tzs = TimeZoneInfo.GetSystemTimeZones().Select(m => new TimeZone
            {
                Id = m.Id,
                DisplayName = m.DisplayName,
                SupportsDaylightSavingTime = m.SupportsDaylightSavingTime
            });

            return this.Success(tzs);
        }

        /// <summary>
        /// To fetch list of log types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of log types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("log-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLogTypesAsync()
        {
            var resources = await this.resourceServices.LogTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of specializations.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of specializations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("specializations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSpecializationsAsync()
        {
            var resources = await this.resourceServices.SpecializationsAsync();
            foreach (var item in resources)
            {
                if (item.OptionalText1 != null)
                {
                    var substring = item.OptionalText1.Substring(0, 9);
                    if (substring == "vs/Icons/")
                    {
                        item.OptionalText1 = item.OptionalText1.Substring(9);
                    }
                }
            }
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of specializations.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of specializations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-specializations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSpecializationsAsync(int? locationId, string? encryptedProviderId)
        {
            int providerId = 0;
            if (encryptedProviderId != "null" && encryptedProviderId != "undefined")
            {
                providerId = Convert.ToInt32(this.aesHelper.Decode(encryptedProviderId));
            }
            var resources = await this.resourceServices.LocationSpecializationsAsync(locationId, providerId);
            return this.Success(resources);
        }
        /// <summary> The fetch lab orders async.</summary>
        /// <returns>The <see cref="Task"/>.</returns>
        [HttpGet]
        [Authorize]
        [Route("lab-order")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabOrdersAsync()
        {
            var resources = await this.resourceServices.LabOrderAsync();
            return this.Success(resources);
        }

        /// <summary> The fetch radiology async.</summary>
        /// <returns>The <see cref="Task"/>.</returns>
        [HttpGet]
        [Authorize]
        [Route("radiology")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRadiologyAsync()
        {
            var resources = await this.resourceServices.RadiologyAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of services.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of services.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("services")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchServicesAsync()
        {
            var resources = await this.resourceServices.ServicesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of insurance companies.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of insurance companies.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("insurance-companies")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInsuranceCompaniesAsync()
        {
            var resources = await this.resourceServices.InsuranceCompaniesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of relationships.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of relationships.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("relationships")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRelationships()
        {
            var resources = await this.resourceServices.Relationships();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of relationships.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of relationships.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("document-categories")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public ActionResult FetchDocumentCategories()
        {
            var resources = this.resourceServices.DocumentCategories();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of provider appointment dates.
        /// </summary>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of provider appointment dates.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("provider-appointment-dates")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderAppointmentDatesAsync(int providerId)
        {
            var resources = await this.resourceServices.ProviderAppointmentDatesAsync(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// /// <param name="isBabyRegistration">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientsAsync(string? searchParam, int? providerId, bool? isBabyRegistration)
        {
            var resources = await this.resourceServices.PatientsAsync(searchParam, providerId, isBabyRegistration);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>To fetch list of patients for scan.</summary>
        /// <param name="searchParam">The search parameter.</param>
        /// <returns>The <see cref="Task"/>.</returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientsScanAsync(string? searchParam)
        {
            var resources = await this.resourceServices.PatientsScanAsync(searchParam);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("appointments")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAppointmentsAsync(string? searchParam)
        {
            var resources = await this.resourceServices.AppointmentsAsync(searchParam);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of departments.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of departments.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("fetch-providerdepartments")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDepartmentsAsync(int? providerId, [FromHeader] LocationHeader header)
        {

            var resources = await this.resourceServices.DepartmentsAsync(providerId, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of departments.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of departments.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("doctor-specialization")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorSpecializationAsync(int? providerId, [FromHeader] LocationHeader header)
        {

            var resources = await this.resourceServices.DoctorSpecializationAsync(providerId, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of surgery types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of surgery types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("surgeryTypes")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSurgeryTypesAsync()
        {
            var resources = await this.resourceServices.SurgeryTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("provider")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderAsync(string searchParam, int? providerId)
        {
            var resources = await this.resourceServices.ProviderAsync(searchParam, providerId);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <param name="isAdmission"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("provider-with-charges")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderWithChargesAsync(string searchParam, bool isAdmission)
        {
            var resources = await this.resourceServices.ProviderWithChargesAsync(searchParam, isAdmission);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of masters.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of masters.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("masters")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMastersAsync(string searchParam)
        {
            var resources = await this.resourceServices.MastersAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of masters.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of masters.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("masters1")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMasters1Async(string searchParam, int? locId)
        {
            var resources = await this.resourceServices.Masters1Async(searchParam, locId);
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of lookups.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookups.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("lookups")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLookupsAsync(string searchParam)
        {
            var resources = await this.resourceServices.LookupsAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of lookup values.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="lookupId">
        /// The lookup Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("lookup-values")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLookupValuesAsync(string searchParam, int? lookupId)
        {
            var resources = await this.resourceServices.LookupValuesAsync(searchParam, lookupId == null || lookupId == 0 ? Convert.ToInt32(lookupId) : (int?)null);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch problem list async.
        /// </summary>
        /// <param name="searchParam">
        /// The search param.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("problem-list")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProblemListAsync(string searchParam)
        {
            var resources = await this.resourceServices.FetchProblemListAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of Practices.
        /// </summary>
        /// <param name="countryId">
        /// The country Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("practice-locations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPracticeLocationsAsync(int countryId)
        {
            var resources = await this.resourceServices.PracticeLocationsAsync(countryId);
            return this.Success(resources);
        }
        [HttpGet]
        [Authorize]
        [Route("practice")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPracticeAsync()
        {
            var resources = await this.resourceServices.PracticeAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of provider locations.
        /// </summary>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of provider locations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("provider-locations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPracticeAndLocation(int providerId, int specializationId, int? locationId)
        {
            //var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.ProviderLocationsAsync(providerId, specializationId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="departmentId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("providers")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviders(int? departmentId, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchProviders(departmentId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("provider-schdule")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ProviderSchdule([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.ProviderSchdule(locationId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-provider")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviders([FromHeader] LocationHeader location)
        {
            int providerId = 0;
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchProviderAsync(locationId, providerId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-provider")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviders(int locationId, string? encryptedProviderId)
        {
            int providerId = 0;
            if (encryptedProviderId != "null" && encryptedProviderId != "undefined")
            {
                providerId = Convert.ToInt32(this.aesHelper.Decode(encryptedProviderId));
            }
            var resources = await this.resourceServices.FetchProviderAsync(locationId, providerId);
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("provider-only")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProvider(int? providerId)
        {
            var resources = await this.resourceServices.FetchProvider(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of drugs.
        /// </summary>
        /// <param name="searchParam">
        /// The search Param.
        /// </param>
        /// <param name="searchValue">
        /// The search Value.
        /// </param>
        /// <param name="medicineSearchType">
        /// The medicine Search Type.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of drugs
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("drugs")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDrugs(string searchParam, string searchValue, string medicineSearchType)
        {
            var resources = await this.resourceServices.DrugsAsync(searchParam, searchValue, medicineSearchType);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch general advice.
        /// </summary>
        /// <param name="providerId">
        /// The provider id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("general-advices")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchGeneralAdvicesAsync(int? providerId, string encryptedAppointmentId)
        {
            if (!string.IsNullOrEmpty(encryptedAppointmentId))
            {
                var appointmentId = Convert.ToInt32(this.aesHelper.Decode(encryptedAppointmentId));
                var appointment = await this.appointmentsServices.FindAsync(appointmentId);
                providerId = appointment.ProviderId;
            }

            var resources = await this.resourceServices.GeneralAdvicesAsync(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch icd.
        /// </summary>
        /// <param name="providerId">
        /// The provider id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("icd-codes")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchICDCodesAsync(int? providerId)
        {
            var resources = await this.resourceServices.ICDCodesAsync(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of ICD Codes.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of ICD Codes.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("master-icd-codes")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchICDCodesAsync(string searchParam)
        {
            var resources = await this.resourceServices.ICDCodesAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of CPT Codes.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of CPT Codes.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("cpt-codes")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCPTCodesAsync(string searchParam)
        {
            var resources = await this.resourceServices.CPTCodesAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch icd.
        /// </summary>
        /// <param name="roleId">
        /// The role id.
        /// </param>
        /// <param name="searchParam">
        /// The search param.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-users")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchUsers(int roleId, string searchParam)
        {
            var resources = await this.resourceServices.FetchBasedOnRole(roleId, searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch icd.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-nurses")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNursesAlt()
        {
            var resources = await this.resourceServices.FetchNurses();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all users except patient user.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-all-users")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllUsers()
        {
            var resources = await this.resourceServices.FetchAllUsers();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch admin and support user async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-admin-super-admin-users")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAdminSuperAdminUsersAsync()
        {
            var resources = await this.resourceServices.FetchAdminSuperAdminUsers();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch lab orders.
        /// </summary>
        /// <param name="providerId">
        /// The provider id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("lab-orders")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabOrdersAsync(int? providerId)
        {
            var resources = await this.resourceServices.LabOrdersAsync(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch radiology orders.
        /// </summary>
        /// <param name="providerId">
        /// The provider id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("radiology-orders")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRadiologyOrdersAsync(int? providerId)
        {
            var resources = await this.resourceServices.RadiologyOrdersAsync(providerId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch general advice.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("languages")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLanguagesAsync()
        {
            var resources = await this.resourceServices.FetchLanguagesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch general advice.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("questions")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchQuestionAsync()
        {
            var resources = await this.resourceServices.FetchQuestionAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch users.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-users-only")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchUsersAsync()
        {
            var resources = await this.resourceServices.FetchUsersAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientAsync()
        {
            var resources = await this.resourceServices.FetchPatientAsync();
            return this.Success(resources);
        }

        ///// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("department-all")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDepartmentAsync()
        {
            var resources = await this.resourceServices.FetchDepartmentAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-pharmacy-categories")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyCategoriesAsync()
        {
            var resources = await this.resourceServices.FetchPharmacyCategoriesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-pharmacy-companies")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyCompaniesAsync()
        {
            var resources = await this.resourceServices.FetchPharmacyCompaniesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-pharmacy-products")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyProductsAsync(int? categoryId, string searchTerm)
        {
            var resources = await this.resourceServices.FetchPharmacyProductsAsync(categoryId, searchTerm);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-pharmacy-product")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyProductAsync(string CategoryName, string searchTerm)
        {
            var resources = await this.resourceServices.FetchPharmacyProductAsync(CategoryName, searchTerm);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch amazon s3 details.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-amazon-details")]
        [ProducesResponseType(200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAmazonS3Async()
        {
            return this.Success(this.amazonS3Configuration);
        }
        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("department")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDepartments()
        {
            var resources = await this.resourceServices.FetchDepartmentAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("providerDepartment")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderDepartments()
        {
            var resources = await this.resourceServices.FetchProviderDepartmentAsync();
            return this.Success(resources);
        }
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("receipt-reports")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReceiptRepoprts(string roleId)
        {
            var resources = await this.resourceServices.FetchReceiptionistReportsAsync(roleId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("account-without-passed-role")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAccountWithoutCurrentRoleAsync(string roleId)
        {
            var resources = await this.resourceServices.FetchRolesAllAsync(roleId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of lookup values.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("instructions")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInstructionsAsync()
        {
            var instructions = await this.resourceServices.FetchInstructionsAsync();

            return this.Success(instructions);
        }

        /// <summary>
        /// To fetch list of lookups.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookups.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("encounter-lookups")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEncounterLookupsAsync(string searchParam)
        {
            var resources = await this.resourceServices.EncounterLookupsAsync(searchParam);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of lookup values.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="lookupId">
        /// The lookup Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("encounter-lookup-values")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEncounterLookupValuesAsync(string searchParam, int? lookupId)
        {
            var resources = lookupId == null || lookupId == 0 ?
                                await this.resourceServices.EncounterLookupValuesAsync(searchParam) :
                                await this.resourceServices.EncounterLookupValuesAsync(searchParam, Convert.ToInt32(lookupId));

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch patient doctors only async.
        /// </summary>
        /// <param name="patientId">
        /// The patient id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("patient-doctors-only")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientDoctorsOnlyAsync(int? patientId, [FromHeader] LocationHeader location)
        {
            if (patientId == null)
            {
                return this.BadRequest();
            }
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;

            var resources = await this.resourceServices.FetchPatientDoctorsAsync(Convert.ToInt32(patientId), locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch patient doctors only async.
        /// </summary>
        /// <param name="patientId">
        /// The patient id.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("patient-appointment-doctors-only")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientAppointmentDoctorsOnlyAsync(int? patientId, [FromHeader] LocationHeader header)
        {
            if (patientId == null)
            {
                return this.BadRequest();
            }

            var resources = await this.resourceServices.FetchPatientAppointmentDoctorsOnlyAsync(Convert.ToInt32(patientId), Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch patient doctors only async.
        /// </summary>
        /// <param name="patientId">
        /// The patient id.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("patient-admission-doctors-only")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientAdmissionDoctorsOnlyAsync(int? patientId, [FromHeader] LocationHeader header)
        {
            if (patientId == null)
            {
                return this.BadRequest();
            }

            var resources = await this.resourceServices.FetchPatientAdmissionDoctorsOnlyAsync(Convert.ToInt32(patientId), Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of pharmacy log types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of pharmacy log types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("pharmacy-log-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmacyLogTypesAsync()
        {
            var resources = await this.resourceServices.PharmacyLogTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of inventory log types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of inventory log types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("inventory-log-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInventoryLogTypesAsync()
        {
            var resources = await this.resourceServices.InventoryLogTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of lab log types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of inventory log types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("lab-log-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabLogTypesAsync()
        {
            var resources = await this.resourceServices.LabLogTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of scan log types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of scan log types.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-log-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanLogTypesAsync()
        {
            var resources = await this.resourceServices.ScanLogTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("discharge-statuses")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDischarge()
        {
            var resources = await this.resourceServices.FetchDischargeStatusAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch lab name.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("lab-name")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabName()
        {
            var resources = await this.resourceServices.FetchLabNames();
            return this.Success(resources);
        }
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("widget-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWidgetTypesAsync()
        {
            try
            {
                var resources = await this.resourceServices.WidgetTypesAsync();
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of widgets.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("widgets")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWidgetsAsync(string? searchParam)
        {
            var resources = await this.resourceServices.WidgetsAsync(searchParam);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// The fetch pay type async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("pay-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPayTypeAsync(bool? isSalucro)
        {
            var resources = await this.resourceServices.FetchPayTypeAsync(isSalucro);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch pay type async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-Pay-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllPayTypesAsync(bool onlyActive)
        {
            var resources = await this.resourceServices.FetchAllPayTypesAsync(onlyActive);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of states.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("states")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchStatesAsync(int id)
        {
            try
            {
                var resources = await this.resourceServices.StatesAsync(id);
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }

        /// <summary>
        /// To fetch list of cities.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("cities")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCitiesAsync(int id)
        {
            try
            {
                var resources = await this.resourceServices.CitiesAsync(id);
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("packages")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPackages(int id)
        {
            var resource = await this.resourceServices.PackageAsync(id);
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of demandProducts.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of demandProducts.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("demand-products")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDemandProducts()
        {
            var resource = await this.resourceServices.DemandProductsAsync();
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("floors")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchFloors([FromHeader] LocationHeader header)
        {
            int locationId = Convert.ToInt32(header.LocationId);
            var resource = await this.resourceServices.FetchFloors(locationId);
            return this.Success(resource);
        }

        /// <summary>
        /// The fetch of all floor names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-floors")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllFloors()
        {
            var resources = await this.resourceServices.FetchAllFloors();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("wards")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWards(int? floorId)
        {
            var resource = await this.resourceServices.FetchWards(floorId);
            return this.Success(resource);
        }

        /// <summary>
        /// The fetch of all wards names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-wards")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllWards(int? floorId)
        {
            var resources = await this.resourceServices.FetchAllWards();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="wardId"></param>
        /// <param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("rooms")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRooms(int? wardId)
        {
            var resource = await this.resourceServices.FetchRooms(wardId);
            return this.Success(resource);
        }

        /// <summary>
        /// The fetch of all rooms names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-rooms")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllRooms(int? wardId, [FromHeader] LocationHeader header)
        {
            int locationid = Convert.ToInt32(header.LocationId);
            var resources = await this.resourceServices.FetchAllRooms(locationid);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="roomId"></param>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("beds")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchBeds([FromBody] int? roomId)
        {
            var resource = await this.resourceServices.FetchBeds(roomId);
            return this.Success(resource);
        }

        /// <summary>
        /// The fetch of all beds names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-beds")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllBeds(int? roomId)
        {
            var resources = await this.resourceServices.FetchAllBeds();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="id">
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("bed-status")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchBedStatus()
        {
            var resource = await this.resourceServices.FetchBedStatus();
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [AllowAnonymous]
        [Route("fetch-beds-filters-data")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchBedsResource([FromBody] BedManagementFilterModel model, [FromHeader] LocationHeader header)
        {
            model.LocationId = Convert.ToInt32(header.LocationId);
            var resource = await this.resourceServices.FetchBedsResource(model);
            return this.Success(resource);
        }
        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="imageId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("image")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchImages(int? imageId)
        {
            var resource = await this.resourceServices.FetchImages(imageId);
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-general-ward-nurses")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNurses()
        {
            var resource = await this.resourceServices.FetchNurses();
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-shifts")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchShifts()
        {
            var resource = await this.resourceServices.FetchShifts();
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [AllowAnonymous]
        [Route("fetch-slots")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSlots([FromBody] int shiftId)
        {
            if (shiftId == 0)
            {
                return this.BadRequest("Please Provide shift details");
            }
            var resource = await this.resourceServices.FetchSlots(shiftId);
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-insurance-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInsuranceTypes()
        {
            var resource = await this.resourceServices.FetchInsuranceTypes();
            return this.Success(resource);
        }

        /// <summary>
        /// To fetch list of packages.
        /// </summary>
        /// <param name="floorId"></param>
        /// The search parameter.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of packages.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-patient-insurances")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientInsurances(int admissionId, int appointmentId)
        {
            if (admissionId == 0 && appointmentId == 0)
            {
                return this.BadRequest();
            }
            if (admissionId > 0 && appointmentId > 0)
            {
                return this.BadRequest();
            }
            var admission = new AdmissionModel();
            var appointment = new AppointmentModel();
            var id = 0;
            if (admissionId > 0)
            {
                admission = await this.admissionService.FindAsync(admissionId);
                if (admission == null)
                {
                    return this.BadRequest();
                }
                id = admission.PatientId;
            }
            if (appointmentId > 0)
            {
                appointment = await this.appointmentsServices.FindAsync(appointmentId);
                if (appointment == null)
                {
                    return this.BadRequest();
                }
                id = appointment.PatientId;
            }
            var records = await this.resourceServices.FetchPatientInsurances(id);
            return this.Success(records);
        }

        /// <summary>
        /// Fetches the retail store.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-retail-store")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRetailStore([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchRetailStoreAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all users except patient user.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-pharma-users")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPharmaUsers()
        {
            var resources = await this.resourceServices.FetchPharmaUsers();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch pharmacy warehouses.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-ware-house")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWareHouse([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchWareHouseAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch iventory warehouses.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-Inventory-warehouse")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInventoryWareHouse([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchInventoryWareHouseAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("meal-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMealTypeAsync()
        {
            var resources = await this.resourceServices.FetchMealTypeAsync();
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of specializations.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of specializations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("provider-practice-location")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderPracticeLocationAsync(int? locationId)
        {
            var resources = await this.resourceServices.ProviderPracticeLocationAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="departmentId"></param>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("admission-providers")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAdmissionProviderAsync(int? departmentId, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.AdmissionProviderAsync(departmentId, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of problems.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of specializations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("problems")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProblemsAsync()
        {
            var resources = await this.resourceServices.FetchProblemsAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch goals.
        /// </summary>
        /// <param name="problemIds">
        /// The problem ids.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("goals")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchGoalsAsync(string problemIds)
        {
            var resources = await this.resourceServices.FetchGoalsAsync(problemIds);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch objectives.
        /// </summary>
        /// <param name="goalIds">
        /// The goal ids.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("objectives")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchObjectivesAsync(string goalIds)
        {
            var resources = await this.resourceServices.FetchObjectivesAsync(goalIds);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch interventions.
        /// </summary>
        /// <param name="objectiveIds">
        /// The objective ids.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("interventions")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInterventionAsync(string objectiveIds)
        {
            var resources = await this.resourceServices.FetchInterventionAsync(objectiveIds);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch shifts async.
        /// </summary>
        /// <param name="locationId">
        /// The location id.
        /// </param>
        /// <returns>
        /// The List of roles. <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("employee-shifts")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> EmployeeShiftsAsync(int? locationId)
        {
            var resources = await this.resourceServices.EmployeeShiftsAsync(locationId);
            return this.Success(resources ?? new List<Resource>());
        }

        /// <summary>
        /// Gets the HWC patients masters asynchronous.
        /// </summary>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("hwc-patients")]
        public async Task<ActionResult> GetHWCPatientsMastersAsync([FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.FetchHwsMastersAsync(int.Parse(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all hwc names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-hwc-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllHWCPatients()
        {
            var resources = await this.resourceServices.FetchAllHWCPatients();
            return this.Success(resources);
        }

        /// <summary>
        /// Gets the HWC patients masters asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("how-did-you-know")]
        public async Task<ActionResult> GetHowDidYouKnowAsync()
        {
            var resources = await this.resourceServices.FetchHowDidYouKnowAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// Gets the HWC patients masters asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("education")]
        public async Task<ActionResult> GetEducationAsync()
        {
            var resources = await this.resourceServices.FetchEducationAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// Gets the HWC patients masters asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("occupation")]
        public async Task<ActionResult> GetOccupationAsync()
        {
            var resources = await this.resourceServices.FetchOccupationAsync();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of dashboard widget count type.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("widget-count-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWidgetCountTypesAsync()
        {
            var resources = await this.resourceServices.WidgetCountTypesAsync();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of dashboard widget count type.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("appointment-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAppointmentTypeAsync()
        {
            var resources = await this.resourceServices.AppointmentTypeAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of referredby names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("referredByNames")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReferredByNamesAsync()
        {
            var resources = await this.resourceServices.ReferredByNamesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the lab departments asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("lab-department")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabDepartmentsAsync()
        {
            var resources = await this.resourceServices.FetchLabDepartmentsAsync();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of scan machine.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-machine-availability")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanMachineeAsync([FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.ScanMachineAsync(int.Parse(header.LocationId));
            return this.Success(resources);
        }


        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of scan machine.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-machine-master")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanMachineMasterAsync(int locationId)
        {
            var resources = await this.resourceServices.FetchScanMachineMasterAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all scan machines.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-scan-machines")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllScanMachines()
        {
            var resources = await this.resourceServices.FetchAllScanMachines();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all scan machines.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-machine-filter-machines")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllScanMachines(int? locationId)
        {
            var resources = await this.resourceServices.FetchScanMachineFilterMachines(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all scan tests.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-scan-tests")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllScanTests()
        {
            var resources = await this.resourceServices.FetchAllScanTests();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all scan tests.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-machine-filter-tests")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanMachineFilterTests(int? locationId)
        {
            var resources = await this.resourceServices.FetchScanMachineFilterTests(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all location scan tests.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("all-location-scan-tests")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationScanTests(int? locationId)
        {
            var resources = await this.resourceServices.FetchLocationScanTests(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="departmentId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-scan-machine-test")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanMachineTestAsync(int? scanMachineMasterId, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FindTestAsync(scanMachineMasterId, locationId);
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("active-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchActivePatientsAsync(string? searchParam, int? providerId)
        {
            var resources = await this.resourceServices.ActivePatientsAsync(searchParam, providerId);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="accountId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("fetch-account")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAccountAsync(int? accountId, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchAccountAsync(accountId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="accountId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("fetch-account-dynamic")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAccountDynamicAsync(string type, string value, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchAccountDynamicAsync(type, value, locationId);
            return this.Success(resources);
        }

        [HttpGet]
        [Route("fetch-provider-account")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderAccountAsync(int? accountId, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchProviderAccountAsync(accountId, locationId);
            return this.Success(resources);
        }

        [HttpGet]
        [Route("fetch-location-scan-provider")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanProviderAccountAsync(int? accountId, int? locationId)
        {
            var resources = await this.resourceServices.FetchScanProviderAccountAsync(accountId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("notification-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNotificationTypes()
        {
            var resources = await this.resourceServices.FetchNotificationTypes();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch radiology orders.
        /// </summary>
        /// <param name="locationId"></param>
        /// <param name="providerId">
        /// The provider id.
        /// </param>
        /// <param name="encryptedProviderId"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("provider-specializations")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderSpecializationAsync(int? providerId, string? encryptedProviderId, int? locationId)
        {
            if (encryptedProviderId != "null" && encryptedProviderId != "undefined")
            {
                providerId = Convert.ToInt32(this.aesHelper.Decode(encryptedProviderId));
            }
            var resources = await this.resourceServices.ProviderSpecializationAsync(providerId, locationId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch inventory prodcut.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Authorize]
        [Route("fetch-inventory-products")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInventoryProduct(int? categoryId, string searchTerm)
        {
            var resources = await this.resourceServices.FetchInventoryProductsAsync(categoryId, searchTerm);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-pay-category")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPayCategory()
        {
            var resources = await this.resourceServices.FetchPayCategory();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("consultation-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchConsultationType()
        {
            var resources = await this.resourceServices.FetchConsultationType();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("availability-charge-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAvailabilityChargeType(int? chargeId, int? providerAvailabilitySlotId)
        {
            var resources = await this.resourceServices.FetchAvailabilityChargeType(chargeId, providerAvailabilitySlotId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of provider locations.
        /// </summary>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <param name="specializationId"></param>
        /// <param name="locationId"></param>
        /// <param name="consultationTypeId"></param>
        /// <param name="appointmentDate"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of provider locations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("provider-availability")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPracticeAndAvailability(int providerId, int specializationId, int? locationId, int? consultationTypeId, string appointmentDate)
        {
            //var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.ProviderAvailabilityAsync(providerId, specializationId, locationId, consultationTypeId, appointmentDate);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of provider locations.
        /// </summary>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <param name="specializationId"></param>
        /// <param name="locationId"></param>
        /// <param name="consultationTypeId"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of provider locations.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("parts-of-day")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPartsOfDay(int providerId, int specializationId, int? locationId, string? startDate, string? endDate)
        {
            //var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.PartsOfDayAsync(providerId, specializationId, locationId, startDate, endDate);
            return this.Success(resources);
        }

        [HttpGet]
        [Authorize]
        [Route("menuAccessedRoles")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMenuAccessedRolesAsync(string? url)
        {
            var resources = await this.resourceServices.MenuAccessedRolesAsync(url);
            return this.Success(resources);
        }

        [HttpPost]
        [Authorize]
        [Route("appointment-patient-filter")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        public async Task<ActionResult> FetchPatientByFilter([FromBody] AppointmentFilterModel model, [FromHeader] LocationHeader header)
        {
            model.LocationId ??= Convert.ToInt32(header.LocationId);
            var response = await this.resourceServices.FetchPatientByFilter(model);
            return this.Success(response);

        }

        /// <summary>
        /// The fetch pay type async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("receipt-area")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReceiptAreaAsync()
        {
            var resources = await this.resourceServices.FetchReceiptAreaAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch pay type async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("receipt-area-id")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReceiptAreaAsyncs()
        {
            var resources = await this.resourceServices.FetchReceiptAreaAsyncs();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of dashboard widget count type.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("active-status")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchActiveStatusAsync()
        {
            var resources = await this.resourceServices.ActiveStatusAsync();
            return this.Success(resources);
        }
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of dashboard widget count type.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("specialization-provider-availability")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSpecializationsForProviderAcailabilityAsync([FromHeader] LocationHeader header)
        {
            int a = Convert.ToInt32(header.LocationId);
            var resources = await this.resourceServices.SpecializationsForProviderAcailabilityAsync(a);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the pathology providers.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpGet]
        [Authorize]
        [Route("pathology-providers")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPathologyProviders([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchPathologyProviders(locationId);
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of countries.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("encounter-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEncounterTypeAsync()
        {
            try
            {
                var resources = await this.resourceServices.EncounterTypeAsync();
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }


        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-scan-test")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationScanTest(int? locationId, int? scanMachineMasterId)
        {
            var resources = await this.resourceServices.FetchLocationScanTestAsync(locationId, scanMachineMasterId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetch scan machines based on location.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-scan-machine")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationScanMachine(int locationId)
        {
            var resources = await this.resourceServices.FetchLocationScanMachineAsync(locationId);
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="scanTestMasterId">
        /// The scan test master Id.
        /// </param>
        /// <param name="scanMachineMasterId">The scan machine master Id</param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-scan-machines-tests")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanMachinesTestsAsync(string? scanTestMasterId, string? scanMachineMasterId)
        {
            var resources = await this.resourceServices.FetchScanMachinesTestsAsync(scanTestMasterId, scanMachineMasterId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all scan classifications.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-scan-classification")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanClassification(int? locationId)
        {
            var resources = await this.resourceServices.FetchScanClassification(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of all scan tests based on scan Classification,subClassification and location.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-sub-classification-tests")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSubClassificationTests(int? locationId, int? classificationId, int? subClassificationId)
        {
            var resources = await this.resourceServices.FetchSubClassificationTests(locationId, classificationId, subClassificationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all scan sub classifications.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("fetch-scan-sub-classification")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanSubClassification(int? classificationId)
        {
            var resources = await this.resourceServices.FetchScanSubClassification(classificationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="departmentId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-Otroom-surgeries")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOTRoomSurgeriesAsync(string? surgeryId, string? otRoomId)
        {
            var resources = await this.resourceServices.FetchOTRoomSurgeriesAsync(surgeryId, otRoomId);
            return this.Success(resources);
        }
        /// <summary>
        /// Fetches the profile image from FTP asynchronous.
        /// </summary>
        /// <param name="imagePath">The image path.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("get-profile-image")]
        public async Task<ActionResult> FetchProfileImageFromFTPAsync(string imagePath)
        {
            if (string.IsNullOrEmpty(imagePath))
            {
                return null;
            }
            if (imagePath == "undefined")
            {
                return null;
            }
            var checkIf = this.ftpUploadHelper.CheckIfFileExistsOnServer(imagePath);
            if (!checkIf)
            {
                return this.BadRequest();
            }
            var memoryStream = await this.ftpUploadHelper.DownloadSmallFilesAsync(imagePath);
            if (memoryStream == null || memoryStream.Length == 0)
            {
                return null;
            }
            var a = File(new System.IO.MemoryStream(memoryStream), "image/jpeg", $@"profile_pic_{DateTime.UtcNow.Ticks}.jpg");
            return a;
        }

        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-scan-test-category-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanTestCategoryTypes(int? scanTestMasterId, int? locationId)
        {
            var resources = await this.resourceServices.FetchScanTestCategoryTypesAsync(scanTestMasterId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of countries.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("locations-practice")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationsForPracticeAsync(string practiceId)
        {
            int id = 0;
            if (!string.IsNullOrEmpty(practiceId))
            {
                id = Convert.ToInt32(this.aesHelper.Decode(practiceId));
            }
            var resources = await this.resourceServices.LocationsForPracticeAsync(id);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of charge category.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of charge category.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("charge-category")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchChargeCategoryAsync()
        {
            var resources = await this.resourceServices.ChargeCategoryAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of practice locations.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("practice-provider-location")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationsForPracticeAsync(int? providerId)
        {
            var resources = await this.resourceServices.FetchLocationsForPracticeAsync(providerId);
            return this.Success(resources);
        }


        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("scan-availability-status")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanAvailabilityStatus()
        {
            var resources = await this.resourceServices.FetchScanAvailabilityStatus();
            return this.Success(resources);
        }



        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("scan-availability-reason")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanAvailabilityReason()
        {
            var resources = await this.resourceServices.FetchScanAvailabilityReason();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch of scan scroll result types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("scan-scroll-result")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanScrollResult()
        {
            var resources = await this.resourceServices.FetchScanScrollResult();
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of chargecategory.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("charge-category-location")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationChargeCategory(int? locationId)
        {
            var resources = await this.resourceServices.FetchLocationChargeCategoryAsync(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// To fetch list of chargecategory.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-case-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchCaseTypes()
        {
            var resources = await this.resourceServices.FetchCaseTypes();
            return this.Success(resources);
        }
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-emergencyCase-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEmergencyCaseTypes()
        {
            var resources = await this.resourceServices.FetchEmergencyCaseTypes();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of scan machine.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("ot-room-master")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOTRoomAsync([FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.FetchOTRoomAsync(int.Parse(header.LocationId));
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of surgery names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of surgery names.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("surgey-names")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> SurgeryNamesAsync()
        {
            var resources = await this.resourceServices.SurgeryNamesAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of surgery names.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of surgery names.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("provider-names")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProvidersForOT([FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchProvidersForOTAsync(locationId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("account-with-passed-role")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAccountWithCurrentRoleAsync(int roleId, [FromHeader] LocationHeader location)
        {
            var locationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var resources = await this.resourceServices.FetchUsersAllAsync(roleId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("anaesthesia-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAnaesthesiaTypesAsync()
        {
            var resources = await this.resourceServices.FetchAnaesthesiaTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch id proofs.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("idproof-names")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchIdProofsAsync()
        {
            var resources = await this.resourceServices.FetchIdProofsAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch rooms.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("fetch-rooms")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRoomAsync()
        {
            var resources = await this.resourceServices.FetchRoomsAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="filter"></param>
        /// <param name="locationId"></param>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <param name="toDay"></param>
        /// The patient model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("patient-name-mobile-Umr")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientNameMobile(string? filter, int locationId, string fromDate, string toDate, string toDay)
        {
            var existingPatients = await this.resourceServices.FetchPatientNameMobileAsync(filter, locationId, fromDate, toDate, toDay);

            return this.Success(existingPatients);
        }

        [HttpGet]
        [AllowAnonymous]
        [Route("patient-name-appointment")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientNameAppointment(string? filter, int locationId, string fromDate, string toDate, string toDay)
        {
            try
            {
                var existingPatients = await this.resourceServices.FetchPatientAppointmentAsync(filter, locationId, fromDate, toDate, toDay);

                return this.Success(existingPatients);

            }

            catch (Exception e)
            {
                return this.Conflict(e.Message + "  :" + fromDate);
            }
        }

        /// <summary>
        /// The fetch patients.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("surgeries")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchsurgerisAsync([FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.FetchsurgerisAsync(int.Parse(header.LocationId));
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-surgeries")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationSurgery(int locationId)
        {
            var resources = await this.resourceServices.FetchLocationSurgeryAsync(locationId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("unique-otRooms")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchUniqueOTRooms(int locationId)
        {
            var resources = await this.resourceServices.FetchUniqueOTRooms(locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch discounts.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("discount-module")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDiscountModule()
        {
            var resources = await this.resourceServices.FetchDiscountModule();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch admission paytypes.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("admission-pay-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAdmissionPayTypes()
        {
            var resources = await this.resourceServices.FetchAdmissionPayTypes();
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch id proofs.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("session-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSessionTypesAsync()
        {
            var resources = await this.resourceServices.FetchSessionTypesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the doctors specialization with all consulations.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("provider-specialization-all-consulation")]
        public async Task<ActionResult> FetchDoctorsSpecializationWithAllConsulations([FromBody] ProviderSpecializationConsultationtypeModel model)
        {
            var resources = await this.resourceServices.FetchProviderSpecializationAllConsulationTypes(model);
            return this.Success(resources);
        }


        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("check-new-patient-appointment")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchNewPatientAppointment(int patientId)
        {

            var resources = await this.resourceServices.FetchNewPatientAppointmentAsync(patientId);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch referral doctor.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("referral-doctor")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReferralDoctorAsync()
        {

            var resources = await this.resourceServices.FetchReferralDoctorAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("in-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchInPatientsAsync(string? searchParam, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.InPatientsAsync(searchParam, Convert.ToInt32(header.LocationId));
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }
            return this.Success(resources);
        }

        /// <summary>
        /// fetch visitTypes
        /// </summary>
        /// <returns></returns>

        [HttpGet]
        [AllowAnonymous]
        [Route("visit-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchVisitTypesAsync()
        {
            var resources = await this.resourceServices.VisitTypesAsync();
            return this.Success(resources);
        }

        [HttpGet]
        [Authorize]
        [Route("tpa-companies")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchTpaCompaniesAsync()
        {
            var resources = await this.resourceServices.FetchTpaCompaniesAsync();
            return this.Success(resources);
        }

        [HttpGet]
        [Authorize]
        [Route("default-chargecategory")]
        [ProducesResponseType(200, Type = typeof(Resource))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDefaultChargeCategoryAsync()
        {
            var resources = await this.resourceServices.FetchDefaultChargeCategoryAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch all notification types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [Route("doctor-week")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorWeek()
        {
            var resources = await this.resourceServices.FetchDoctorWeek();
            return this.Success(resources);
        }

        [HttpGet]
        [AllowAnonymous]
        [Route("out-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchOutPatientsAsync(string? searchParam, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.OutPatientsAsync(searchParam, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }
        [HttpGet]
        [AllowAnonymous]
        [Route("health-card")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchHealthCardAsync()
        {
            var resources = await this.resourceServices.FetchHealthCardAsync();
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch provider.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("unique-surgeries")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSurgery(int locationId)
        {

            var resources = await this.resourceServices.FetchSurgeryAsync(locationId);
            return this.Success(resources);
        }

        [HttpGet]
        [AllowAnonymous]
        [Route("ambulances")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAmbulanceAsync(string? searchParam, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.AmbulancesAsync(searchParam, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        [HttpGet]
        [AllowAnonymous]
        [Route("driver-details")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDriverDetailAsync(string? searchParam, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.DriverDetailAsync(searchParam);
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch reason.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("reasons")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchReasons()
        {
            var resources = await this.resourceServices.FetchReasons();
            return this.Success(resources);
        }
        /// <summary>
        /// The fetch authority.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("authority")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAuthority()
        {
            var resources = await this.resourceServices.FetchAuthorityMasters();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch authority.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("doctorUnit")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorUnit(int? locationId)
        {
            var resources = await this.resourceServices.FetchDoctorUnit(locationId);
            return this.Success(resources);
        }


        /// <summary>
        /// The fetch providers.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("provider-signature")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderSignature()
        {
            var resources = await this.resourceServices.FetchProviderSignature();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch providers.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("consultant-doctors")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchConsultDoctors(string? searchParam, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.FetchConsultDoctors(searchParam, Convert.ToInt32(header.LocationId));
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the med frequencies asynchronous.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("medication-frequency")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMedFrequenciesAsync(string type)
        {
            var resources = await this.resourceServices.FetchMedicationFrequencyAsync(type);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the lookup values based on lookup asynchronous.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("lookup-value-on-name")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLookupValuesBasedOnLookupAsync(string name)
        {
            var resources = await this.resourceServices.LookupValuesBasedOnLookupNameAsync(name);
            return this.Success(resources);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="searchParam"></param>
        /// <param name="fromDate"></param>
        /// <param name="toDate"></param>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("registered-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRegisteredPatients(string? searchParam, string? fromDate, string? toDate)
        {
            var resources = await this.resourceServices.FetchRegisteredPatientsAsync(searchParam, fromDate, toDate);
            foreach (var resource in resources)
            {
                resource.EncryptedId = this.aesHelper.Encode(resource.Id.ToString());
            }

            return this.Success(resources);
        }

        [HttpGet]
        [Route("fetch-provider-location")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProviderLocationsAsync(string? encryptedProviderId, int? locationId)
        {
            int? providerId = !string.IsNullOrEmpty(encryptedProviderId) ? Convert.ToInt32(aesHelper.Decode(encryptedProviderId)) : Convert.ToInt32(string.Empty);

            var resources = await this.resourceServices.FetchProviderLocationsAsync(providerId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the lookup values based on lookup asynchronous.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("specialization-provider")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchSpecializationProviderAsync(int? specializationId)
        {
            var resources = await this.resourceServices.FetchSpecializationProviderAsync(specializationId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the scan distinct out patients asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-distinct-out-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ScanDistinctOutPatients(string fromDate, string toDate, int locationId)
        {
            var resources = await this.resourceServices.ScanDistinctOutPatients(fromDate, toDate, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the scan bulk reschedue patients      
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-bulk-reschedue-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ScanBulkRCPatients(string fromDate, string toDate, int locationId)
        {
            var resources = await this.resourceServices.ScanBulkRCPatients(fromDate, toDate, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the packages.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("package-modules")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> PackageModules(int locationId, string packageType, string moduleType)
        {
            var resources = await this.resourceServices.PackageModules(locationId, packageType, moduleType);
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch package types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("counselling-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> CounsellingTypes()
        {
            var resources = await this.resourceServices.CounsellingTypes();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch package types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("package-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> PackageTypes()
        {
            var resources = await this.resourceServices.PackageTypes();
            return this.Success(resources);
        }

        /// <summary>
        /// The fetch module types.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("module-types")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModuleTypes()
        {
            var resources = await this.resourceServices.ModuleTypes();
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the scan distinct out patients asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("diet-guidlines")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DietGuidlines(int? typeId)
        {
            var resources = await this.resourceServices.DietGuidlines(typeId);
            return this.Success(resources);
        }
        /// <summary>
        /// Fetches the scan distinct out patients asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [AllowAnonymous]
        [Route("diet-type")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DietType()
        {
            var resources = await this.resourceServices.DietTypes();
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of patients.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="providerId">
        /// The provider Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("doctor-appointment-notice")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDoctorAppointmentNoticeAsync(int? locationId, int? providerId, string? appointmentDate)
        {
            var resources = await this.resourceServices.DoctorAppointmentNoticeAsync(locationId, providerId, appointmentDate);

            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of Scan Appointment Notice.
        /// </summary>
        /// <param name="locationId">
        /// The location Id.
        /// </param>
        /// <param name="machineId">
        /// The machine Id.
        /// </param>
        /// <param name="appointmentDate">
        /// The appointment Date.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of patients.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("scan-appointment-notice")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanAppointmentNoticeAsync(int? locationId, int? machineId, string appointmentDate)
        {
            var resources = await this.resourceServices.ScanAppointmentNoticeAsync(locationId, machineId, appointmentDate);
            return this.Success(resources);
        }
        [HttpGet]
        [AllowAnonymous]
        [Route("emergency-detail")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEmergencyDetailAsync(int? emergencyTypeId)
        {
            var resources = await this.resourceServices.FetchEmergencyDetailAsync(emergencyTypeId);
            return this.Success(resources);
        }

        [HttpGet]
        [Route("lab-vacutainer")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabVacutainersAsync()
        {
            var resources = await this.resourceServices.FetchLabVacutainersAsync();
            return this.Success(resources);
        }

        [HttpGet]
        [Route("get-anc-number")]
        [ProducesResponseType(200, Type = typeof(Resource))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAncNumberAsync(int patientId)
        {
            var resources = await this.resourceServices.FetchAncNumberAsync(patientId);
            return this.Success(resources);
        }

        [HttpGet]
        [Route("get-patient-bmi")]
        [ProducesResponseType(200, Type = typeof(Resource))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientBMI(int patientId)
        {
            var resources = await this.resourceServices.FetchPatientBMI(patientId);
            return this.Success(resources);
        }


        /// <summary>
        /// To fetch list of countries.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of countries.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("patient-registration-charge")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchPatientRegistrationChargeAsync(int locationId)
        {
            try
            {
                var resources = await this.resourceServices.PatientRegistrationChargeAsync(locationId);
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }

        /// <summary>
        /// To fetch list of lookup values.
        /// </summary>       
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("encounter-lookup")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEncounterLookUp()
        {
            var resources = await this.resourceServices.FetchEncounterLookup();
            return this.Success(resources);
        }
        /// <summary>
        /// To fetch list of lookup values.
        /// </summary>
        /// <param name="searchParam">
        /// The search parameter.
        /// </param>
        /// <param name="lookupId">
        /// The lookup Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of Lookup values.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("encounterlookupvalues")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchEncounterLookupValues(int lookupId)
        {
            var resources = await this.resourceServices.FetchEncounterLookUpValues(lookupId);
            return this.Success(resources);
        }

        /// <summary>To fetch scan machines based on test.</summary>
        /// <param name="locationId"></param>
        /// <param name="scanTestMasterId"></param>
        /// <returns>The <see cref="Task"/>.</returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-scan-test-machines")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanTestMachinesAsync(int? locationId, int? scanTestMasterId)
        {
            var resources = await this.resourceServices.FetchScanTestMachinesAsync(locationId, scanTestMasterId);
            return this.Success(resources);
        }
        /// <summary>To fetch scan machines based on test.</summary>
        /// <param name="locationId"></param>
        /// <param name="searchParam"></param>
        /// <returns>The <see cref="Task"/>.</returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-admission-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchScanTestMachinesAsync(string? searchParam, int? locationId)
        {
            var resources = await this.resourceServices.FetchAdmissionPatients(searchParam, locationId);
            return this.Success(resources);
        }

        /// <summary>To fetch scan machines based on test.</summary>
        /// <param name="parentPatientId"></param>
        /// <returns>The <see cref="Task"/>.</returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("fetch-parent-patients")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchParentPatients(int? parentPatientId)
        {
            var resources = await this.resourceServices.FetchParentPatients(parentPatientId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the lab masters asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("lab-master")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabMastersAsync()
        {
            var resources = await this.resourceServices.FetchLabMastersAsync();
            return this.Success(resources);
        }

        [HttpGet]
        [Route("location-warehouses")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchWareHousesAsync(int id)
        {
            try
            {
                var resources = await this.resourceServices.FetchWareHousesAsync(id);
                return this.Success(resources);
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                throw;
            }
        }

        [HttpGet]
        [Route("warehouse-retail")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchRetailsAsync(int wId)
        {


            var resources = await this.resourceServices.FetchRetailsAsync(wId);
            return this.Success(resources);


        }

        [HttpGet]
        [Route("fetch-template-header")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchTemplatesAsync()

        {

            var resources = await this.resourceServices.FetchTemplatesAsync();
            return this.Success(resources);

        }
        [HttpGet]
        [Route("fetch-modules-master")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchModulesMasterAsync()

        {

            var resources = await this.resourceServices.FetchModulesMasterAsync();
            return this.Success(resources);

        }


        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("fetch-all-labs")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllLabAsync()

        {
                var resources = await this.resourceServices.FetchAllLabAsync();
                return this.Success(resources);
            
        }
        /// <summary>
        /// To fetch list of providers.
        /// </summary>
        /// <param name="departmentId">
        /// The provider Id.
        /// </param>
        /// <param name="location"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of providers.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Authorize]
        [Route("providers-location")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchProvidersLocation(int? departmentId, int? locationId)
        {            
            var resources = await this.resourceServices.FetchProvidersLocation(departmentId, locationId);
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches the lab external agencies asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("lab-external-agencies")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLabExternalAgenciesAsync()
        {
            var resources = await this.resourceServices.FetchAllLabExternalAgenciesAsync();
            return this.Success(resources);
        }

        /// <summary>
        /// Fetches all active lab external agencies asynchronous.
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [Route("lab-external-agencies-active")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAllActiveLabExternalAgenciesAsync()
        {
            var resources = await this.resourceServices.FetchAllActiveLabExternalAgenciesAsync();
            return this.Success(resources);
        }

        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Returns list of scan machine.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [AllowAnonymous]
        [Route("location-charge-template")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchLocationChargeTemplateAsync(int? admissionPayTypeId, [FromHeader] LocationHeader header)
        {
            var resources = await this.resourceServices.FetchLocationChargeTemplateAsync(admissionPayTypeId, int.Parse(header.LocationId));
            return this.Success(resources);
        }
        
        /// <summary>
         /// The fetch diet conditions.
         /// </summary>
         /// <returns>
         /// The <see cref="Task"/>.
         /// </returns>
        [HttpGet]
        [Route("diet-conditions")]
        [ProducesResponseType(200, Type = typeof(List<Resource>))]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchDietConditions()
        {
            var resources = await this.resourceServices.FetchDietConditions();
            return this.Success(resources);
        }

    }
}


    

    
